/*--------------------------------------------------------------------------*\

    FILE....: PORTUSER.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 24/8/00

    User mode low level port I/O module.  Used for O/S's where port I/O can
    be performed directly from user mode.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

				 INCLUDES

\*-------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>

#include "port.h"
#include "ioports.h"

static USHORT base[MAX_VPB];
static USHORT boards;

/*-------------------------------------------------------------------------*\

				 FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::PortUser
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Initialises PortUser object.

\*-------------------------------------------------------------------------*/

PortUser::PortUser() {
	// attempt to gain access to ports (must be root)
	assert(iopriv(1) == 0);
	boards = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortUser::PortUser
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Frees PortUser.

\*-------------------------------------------------------------------------*/

PortUser::~PortUser() {
	iopriv(0);
	boards = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::addBoard
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Allocates the operating resources required to access a VPB.

\*-------------------------------------------------------------------------*/

void PortUser::addBoard(USHORT new_base, USHORT span)
//  USHORT new_base;	base address of port block
//  USHORT span;	number of ports in block
{
	assert(boards <MAX_VPB);
	base[boards++] = new_base;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::write
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Writes a single 16-bit word to a 16-bit I/O port.

\*-------------------------------------------------------------------------*/

void PortUser::write(USHORT board, USHORT offset, word value)
//      USHORT	board;	VPB number
//      USHORT  offset;	Offset of port from base address of board
//      word	value;	value to write
{
	assert(board < boards);

	USHORT port = base[board] + offset;
	ioset(port, value);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::blockWrite
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Writes a block of 16-bit words to a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortUser::blockWrite(USHORT board, USHORT offset, USHORT length, 
			  word *buf)
//      USHORT	board;	VPB number
//      USHORT  offset;	Offset of port from base address of board
//      USHORT  length;	number of words (not butes) in buf
//      word	buf[];	buffer to write to VPB
{

	/* validate arguments */

	assert(board < boards);
	assert(buf != NULL);
	assert(length < MAX_LENGTH);

	/* perform block write */

	USHORT port = base[board] + offset;
	iowrite(port, buf, length);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser:blockRead
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Reads a block of 16-bit words from a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortUser::blockRead(USHORT board, USHORT offset, USHORT length, word *buf)
//      USHORT	board;	VPB number
//      USHORT  offset;	Offset of port from base address of board
//      USHORT  length;	number of words (not butes) in buf
//      word	buf[];	buffer read from VPB
{

	/* validate arguments */

	assert(board < boards);
	assert(buf != NULL);
	assert(length < MAX_LENGTH);

	/* perform block read */

	USHORT port = base[board] + offset;
	ioread(port, buf, length);	
}

